<?php

namespace App\Services;

use App\Models\Account;
use App\Models\LedgerEntry;

class AccountingService
{
  public function postInvoice(
    int $invoiceId,
    float $salesNet,     // after discount, before VAT
    float $vatAmount,
    float $cogs,
    float $paidAmount,
    float $balance,
    string $paymentMethod
  ): void {
    $cash = Account::where('code','1001')->firstOrFail();
    $bank = Account::where('code','1002')->firstOrFail();
    $ar   = Account::where('code','1004')->firstOrFail();

    $sales = Account::where('code','4001')->firstOrFail();
    $vatPayable = Account::where('code','2001')->firstOrFail();

    $cogsAcc = Account::where('code','5001')->firstOrFail();
    $invAcc  = Account::where('code','1003')->firstOrFail();

    // 1) Revenue/VAT postings: Debit Cash/Bank/AR, Credit Sales + VAT
    // paid portion -> Cash/Bank, balance -> AR
    if ($paidAmount > 0) {
      $assetAcc = ($paymentMethod === 'transfer' || $paymentMethod === 'pos') ? $bank : $cash;
      $this->entry($assetAcc->id, $paidAmount, 0, 'invoice', $invoiceId, 'Invoice payment received');
    }
    if ($balance > 0) {
      $this->entry($ar->id, $balance, 0, 'invoice', $invoiceId, 'Invoice balance (customer owes)');
    }

    $this->entry($sales->id, 0, $salesNet, 'invoice', $invoiceId, 'Sales revenue');
    if ($vatAmount > 0) {
      $this->entry($vatPayable->id, 0, $vatAmount, 'invoice', $invoiceId, 'VAT payable');
    }

    // 2) COGS: Debit COGS, Credit Inventory
    if ($cogs > 0) {
      $this->entry($cogsAcc->id, $cogs, 0, 'invoice', $invoiceId, 'COGS');
      $this->entry($invAcc->id, 0, $cogs, 'invoice', $invoiceId, 'Inventory reduction');
    }
  }

  public function postExpense(int $expenseId, float $amount, int $expenseAccountId, int $paymentAccountId, string $note=''): void
  {
    $this->entry($expenseAccountId, $amount, 0, 'expense', $expenseId, $note ?: 'Expense');
    $this->entry($paymentAccountId, 0, $amount, 'expense', $expenseId, $note ?: 'Expense paid');
  }

  private function entry(int $accountId, float $debit, float $credit, string $refType, int $refId, string $desc): void
  {
    LedgerEntry::create([
      'account_id' => $accountId,
      'debit' => round($debit, 2),
      'credit' => round($credit, 2),
      'reference_type' => $refType,
      'reference_id' => $refId,
      'description' => $desc,
    ]);
  }
}
