<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Product;
use App\Models\ProductItem;
use App\Models\StockMovement;
use App\Services\AccountingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class InvoiceController extends Controller
{
  public function index(Request $request)
  {
    $q = Invoice::query()->orderByDesc('created_at');
    if ($request->query('status')) {
      $q->where('status', $request->query('status'));
    }
    return response()->json($q->limit(200)->get());
  }

  public function show(Invoice $invoice)
  {
    $invoice->load('items');
    return response()->json($invoice);
  }

  public function update(Request $request, Invoice $invoice)
  {
    $data = $request->validate([
      'bill_to_name' => ['nullable','string'],
      'bill_to_address' => ['nullable','string'],
      'payment_bank' => ['nullable','string'],
      'payment_name' => ['nullable','string'],
      'payment_account' => ['nullable','string'],
      'payment_method' => ['nullable','in:cash,transfer,pos,mixed'],
    ]);

    $invoice->update($data);
    return response()->json($invoice);
  }

  public function destroy(Invoice $invoice)
  {
    $invoice->items()->delete();
    $invoice->delete();
    return response()->json(['status' => 'deleted']);
  }

  public function applyPayment(Request $request, Invoice $invoice, AccountingService $acct)
  {
    $data = $request->validate([
      'amount' => ['required','numeric','min:0.01'],
      'payment_method' => ['nullable','in:cash,transfer,pos,mixed'],
    ]);

    if ($invoice->status === 'void') {
      return response()->json(['message' => 'Invoice void'], 422);
    }

    $amount = min((float)$data['amount'], (float)$invoice->balance);
    if ($amount <= 0) {
      return response()->json(['message' => 'No balance due'], 422);
    }

    $invoice->paid_amount = round($invoice->paid_amount + $amount, 2);
    $invoice->balance = round(max($invoice->total - $invoice->paid_amount, 0), 2);
    $invoice->status = $invoice->balance > 0 ? 'partial' : 'paid';
    if (!empty($data['payment_method'])) {
      $invoice->payment_method = $data['payment_method'];
    }
    $invoice->save();

    $acct->postInvoice(
      $invoice->id,
      0,
      0,
      0,
      $amount,
      $invoice->balance,
      $invoice->payment_method
    );

    return response()->json($invoice);
  }

  public function store(Request $request, AccountingService $acct)
  {
    \assertOpenPeriod(now());

    $data = $request->validate([
      'customer_id' => ['nullable','integer','exists:customers,id'],
      'bill_to_name' => ['nullable','string'],
      'bill_to_address' => ['nullable','string'],
      'payment_bank' => ['nullable','string'],
      'payment_name' => ['nullable','string'],
      'payment_account' => ['nullable','string'],
      'payment_method' => ['required','in:cash,transfer,pos,mixed'],
      'discount_type' => ['required','in:none,percent,amount'],
      'discount_value' => ['numeric','min:0'],
      'vat_percent' => ['numeric','min:0'],
      'paid_amount' => ['numeric','min:0'],
      'client_ref' => ['nullable','string'],
      'items' => ['required','array','min:1'],
      'items.*.product_id' => ['nullable','integer','exists:products,id'],
      'items.*.item_name' => ['nullable','string'],
      'items.*.qty' => ['required','integer','min:1'],
      'items.*.unit_price' => ['required','numeric','min:0'],
      'items.*.serial_numbers' => ['nullable','array'],
      'items.*.serial_numbers.*' => ['string'],
    ]);

    $clientRef = $request->input('client_ref');
    if ($clientRef) {
      $existing = Invoice::where('client_ref', $clientRef)->first();
      if ($existing) {
        return response()->json([
          'invoice_id' => $existing->id,
          'invoice_no' => $existing->invoice_no,
          'status' => $existing->status,
          'total' => $existing->total,
          'paid' => $existing->paid_amount,
          'balance' => $existing->balance,
          'duplicate' => true,
        ], 200);
      }
    }

    return DB::transaction(function () use ($data, $acct, $clientRef) {
      // Lock products to prevent oversell
      $productIds = collect($data['items'])->pluck('product_id')->unique()->values()->all();
      $products = Product::whereIn('id', $productIds)->lockForUpdate()->get()->keyBy('id');

      $subtotal = 0;
      $cogs = 0;

      foreach ($data['items'] as $it) {
        $p = null;
        if (!empty($it['product_id'])) {
          $p = $products[$it['product_id']];
          if (!$p->is_active) abort(422, 'Product inactive: '.$p->name);
          if ($p->stock_qty < $it['qty']) abort(422, 'Insufficient stock for: '.$p->name);
          if (!empty($it['serial_numbers'])) {
            $serials = array_values(array_unique(array_map(fn($s) => trim($s), $it['serial_numbers'])));
            $serials = array_values(array_filter($serials, fn($s) => $s !== ''));
            if (count($serials) !== (int)$it['qty']) {
              abort(422, 'Serial count must match qty for: '.$p->name);
            }
            $available = ProductItem::where('product_id', $p->id)
              ->whereIn('serial_number', $serials)
              ->where('status', 'in_stock')
              ->count();
            if ($available !== count($serials)) {
              abort(422, 'One or more serials not in stock for: '.$p->name);
            }
          }
        }

        $subtotal += $it['unit_price'] * $it['qty'];
        if ($p) {
          $cogs += $p->cost_price * $it['qty'];
        }
      }

      // Discount
      $discountAmount = 0;
      if ($data['discount_type'] === 'percent') {
        $discountAmount = $subtotal * ((float)$data['discount_value'] / 100.0);
      } elseif ($data['discount_type'] === 'amount') {
        $discountAmount = (float)$data['discount_value'];
      }
      $discountAmount = max(min($discountAmount, $subtotal), 0);

      $salesNet = max($subtotal - $discountAmount, 0);
      $vatPercent = (float)($data['vat_percent'] ?? 7.5);
      $vatAmount = $salesNet * ($vatPercent / 100.0);

      $total = $salesNet + $vatAmount;

      $paid = min((float)$data['paid_amount'], $total);
      $balance = max($total - $paid, 0);

      $status = $balance > 0 ? ($paid > 0 ? 'partial' : 'unpaid') : 'paid';

      $invoice = Invoice::create([
        'invoice_no' => $this->nextInvoiceNo(),
        'customer_id' => $data['customer_id'] ?? null,
        'subtotal' => round($subtotal, 2),
        'discount_type' => $data['discount_type'],
        'discount_value' => (float)$data['discount_value'],
        'discount_amount' => round($discountAmount, 2),
        'vat_percent' => round($vatPercent, 2),
        'vat_amount' => round($vatAmount, 2),
        'total' => round($total, 2),
        'paid_amount' => round($paid, 2),
        'balance' => round($balance, 2),
        'payment_method' => $data['payment_method'],
        'status' => $status,
        'client_ref' => $clientRef,
        'issued_by' => auth()->id(),
        'bill_to_name' => $data['bill_to_name'] ?? null,
        'bill_to_address' => $data['bill_to_address'] ?? null,
        'payment_bank' => $data['payment_bank'] ?? null,
        'payment_name' => $data['payment_name'] ?? null,
        'payment_account' => $data['payment_account'] ?? null,
      ]);

      // items + stock movements
      foreach ($data['items'] as $it) {
        $p = null;
        if (!empty($it['product_id'])) {
          $p = $products[$it['product_id']];
        }

        $lineTotal = ($it['unit_price'] * $it['qty']);

        InvoiceItem::create([
          'invoice_id' => $invoice->id,
          'product_id' => $p?->id,
          'item_name' => $p?->name ?? ($it['item_name'] ?? 'Item'),
          'sku' => $p?->sku,
          'barcode' => $p?->barcode,
          'unit_price' => round($it['unit_price'], 2),
          'cost_price' => round($p?->cost_price ?? 0, 2),
          'qty' => (int)$it['qty'],
          'line_discount' => 0,
          'line_total' => round($lineTotal, 2),
        ]);

        if ($p) {
          $p->decrement('stock_qty', (int)$it['qty']);

          if (!empty($it['serial_numbers'])) {
            $serials = array_values(array_unique(array_map(fn($s) => trim($s), $it['serial_numbers'])));
            $serials = array_values(array_filter($serials, fn($s) => $s !== ''));
            ProductItem::where('product_id', $p->id)
              ->whereIn('serial_number', $serials)
              ->update(['status' => 'sold']);
          }

          StockMovement::create([
            'product_id' => $p->id,
            'type' => 'sale',
            'qty_change' => -((int)$it['qty']),
            'reference_type' => 'invoice',
            'reference_id' => $invoice->id,
            'note' => 'Sold via POS',
          ]);
        }
      }

      // Accounting postings
      $acct->postInvoice(
        $invoice->id,
        (float)$salesNet,
        (float)$vatAmount,
        (float)$cogs,
        (float)$paid,
        (float)$balance,
        $data['payment_method']
      );

      return response()->json([
        'invoice_id' => $invoice->id,
        'invoice_no' => $invoice->invoice_no,
        'status' => $invoice->status,
        'total' => $invoice->total,
        'paid' => $invoice->paid_amount,
        'balance' => $invoice->balance,
      ], 201);
    });
  }

  private function nextInvoiceNo(): string
  {
    $year = now()->format('Y');
    $last = Invoice::whereYear('created_at', $year)->orderByDesc('id')->first();
    $seq = 1;
    if ($last) {
      // INV-2026-000001
      $parts = explode('-', $last->invoice_no);
      $seq = isset($parts[2]) ? ((int)$parts[2] + 1) : ($last->id + 1);
    }
    return sprintf('INV-%s-%06d', $year, $seq);
  }
}
